<?php
/**
 * پارسر HTML برای تلگرام
 * HTML Parser for Telegram with formatting support
 */

class HTMLParser {
    
    /**
     * تبدیل متن به فرمت HTML تلگرام
     */
    public static function parse($text, $entities = []) {
        if (empty($entities)) {
            return self::escapeHTML($text);
        }
        
        // مرتب‌سازی entities بر اساس offset
        usort($entities, function($a, $b) {
            return $a['offset'] - $b['offset'];
        });
        
        $result = '';
        $lastOffset = 0;
        
        foreach ($entities as $entity) {
            $offset = $entity['offset'];
            $length = $entity['length'];
            $type = $entity['type'];
            
            // اضافه کردن متن قبل از entity
            $result .= self::escapeHTML(mb_substr($text, $lastOffset, $offset - $lastOffset));
            
            // گرفتن متن entity
            $entityText = mb_substr($text, $offset, $length);
            
            // فرمت‌بندی بر اساس نوع
            switch ($type) {
                case 'bold':
                    $result .= '<b>' . self::escapeHTML($entityText) . '</b>';
                    break;
                    
                case 'italic':
                    $result .= '<i>' . self::escapeHTML($entityText) . '</i>';
                    break;
                    
                case 'code':
                    $result .= '<code>' . self::escapeHTML($entityText) . '</code>';
                    break;
                    
                case 'pre':
                    $result .= '<pre>' . self::escapeHTML($entityText) . '</pre>';
                    break;
                    
                case 'text_link':
                    $url = isset($entity['url']) ? $entity['url'] : '';
                    $result .= '<a href="' . self::escapeHTML($url) . '">' . self::escapeHTML($entityText) . '</a>';
                    break;
                    
                case 'url':
                    $result .= '<a href="' . self::escapeHTML($entityText) . '">' . self::escapeHTML($entityText) . '</a>';
                    break;
                    
                case 'mention':
                case 'hashtag':
                case 'email':
                    $result .= self::escapeHTML($entityText);
                    break;
                    
                default:
                    $result .= self::escapeHTML($entityText);
            }
            
            $lastOffset = $offset + $length;
        }
        
        // اضافه کردن بقیه متن
        $result .= self::escapeHTML(mb_substr($text, $lastOffset));
        
        return $result;
    }
    
    /**
     * ساخت blockquote
     */
    public static function blockquote($text) {
        return "<blockquote>" . self::escapeHTML($text) . "</blockquote>";
    }
    
    /**
     * ساخت پیام با فرمت
     */
    public static function format($text, $bold = false, $italic = false, $code = false) {
        $result = $text;
        
        if ($code) {
            $result = '<code>' . self::escapeHTML($result) . '</code>';
        }
        
        if ($bold) {
            $result = '<b>' . self::escapeHTML($result) . '</b>';
        }
        
        if ($italic) {
            $result = '<i>' . self::escapeHTML($result) . '</i>';
        }
        
        if (!$code && !$bold && !$italic) {
            $result = self::escapeHTML($result);
        }
        
        return $result;
    }
    
    /**
     * Escape کردن کاراکترهای خاص HTML
     */
    public static function escapeHTML($text) {
        return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
    }
    
    /**
     * ساخت لینک
     */
    public static function link($text, $url) {
        return '<a href="' . self::escapeHTML($url) . '">' . self::escapeHTML($text) . '</a>';
    }
    
    /**
     * فرمت کردن سایز فایل
     */
    public static function formatFileSize($bytes) {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }
    
    /**
     * نمایش ستاره‌های امتیاز
     */
    public static function stars($rating, $maxRating = 5) {
        $fullStar = '⭐';
        $emptyStar = '☆';
        
        $stars = '';
        for ($i = 1; $i <= $maxRating; $i++) {
            if ($i <= $rating) {
                $stars .= $fullStar;
            } else {
                $stars .= $emptyStar;
            }
        }
        
        return $stars;
    }
    
    /**
     * نمایش امتیاز با ستاره و عدد
     */
    public static function ratingDisplay($averageRating, $totalRatings) {
        if ($totalRatings == 0) {
            return '☆☆☆☆☆ (بدون امتیاز)';
        }
        
        $stars = self::stars(round($averageRating));
        return $stars . " {$averageRating}/5 ({$totalRatings} رأی)";
    }
    
    /**
     * ساخت پیام خوش‌آمدگویی با فرمت زیبا
     */
    public static function welcomeMessage($firstName) {
        $message = "سلام " . self::format($firstName, true) . " عزیز! 👋\n\n";
        $message .= self::blockquote("به ربات آپلودر خوش آمدید\nبرای دانلود فایل، لینک دریافتی را ارسال کنید.");
        return $message;
    }
    
    /**
     * پیام اطلاعات فایل
     */
    public static function fileInfo($fileName, $fileSize, $downloadCount) {
        $message = "📁 " . self::format("اطلاعات فایل", true) . "\n\n";
        $message .= "📌 نام: " . self::format($fileName, false, false, true) . "\n";
        $message .= "💾 حجم: " . self::format(self::formatFileSize($fileSize)) . "\n";
        $message .= "📥 تعداد دانلود: " . self::format($downloadCount) . "\n\n";
        $message .= self::blockquote("در حال ارسال فایل...");
        return $message;
    }
}
