<?php
/**
 * هندلر پنل کاربران
 * User Panel Handler
 */

/**
 * نمایش اطلاعات کاربر
 */
function handleUserInfo($chatId, $userId) {
    $user = Database::getUser($userId);
    
    if (!$user) {
        Telegram::sendMessage($chatId, "❌ اطلاعات یافت نشد!");
        return;
    }
    
    $stats = Database::getUserDownloadStats($userId);
    
    $message = "👤 " . HTMLParser::format("اطلاعات شما", true) . "\n\n";
    
    $userInfo = "🆔 " . HTMLParser::format("شناسه: ", true) . $userId . "\n";
    $userInfo .= "👤 " . HTMLParser::format("نام: ", true) . ($user['first_name'] ?: 'ندارد');
    if ($user['last_name']) {
        $userInfo .= " " . $user['last_name'];
    }
    $userInfo .= "\n";
    
    if ($user['username']) {
        $userInfo .= "📱 " . HTMLParser::format("یوزرنیم: ", true) . "@" . $user['username'] . "\n";
    }
    
    $userInfo .= "📅 " . HTMLParser::format("تاریخ عضویت: ", true) . date('Y/m/d', strtotime($user['join_date'])) . "\n";
    $userInfo .= "📥 " . HTMLParser::format("تعداد دانلود: ", true) . number_format($stats['download_count']) . "\n";
    $userInfo .= "💾 " . HTMLParser::format("حجم دانلود: ", true) . HTMLParser::formatFileSize($stats['total_size']);
    
    $message .= HTMLParser::blockquote($userInfo);
    
    Telegram::sendMessage($chatId, $message, Telegram::getUserKeyboard());
}

/**
 * نمایش لیست فایل‌های دانلود شده
 */
function handleUserDownloadedFiles($chatId, $userId) {
    $files = Database::getUserDownloadedFiles($userId);
    
    if (empty($files)) {
        $message = "📥 " . HTMLParser::format("فایل‌های دانلود شده", true) . "\n\n";
        $message .= HTMLParser::blockquote("شما هنوز فایلی دانلود نکرده‌اید.");
        
        Telegram::sendMessage($chatId, $message, Telegram::getUserKeyboard());
        return;
    }
    
    $message = "📥 " . HTMLParser::format("فایل‌های دانلود شده شما", true) . "\n\n";
    $message .= HTMLParser::blockquote("تعداد: " . count($files) . " فایل");
    $message .= "\n\n";
    
    // ساخت دکمه‌های Inline برای هر فایل
    $buttons = [];
    $counter = 1;
    
    foreach ($files as $file) {
        // نمایش تایتل اگر وجود دارد، در غیر این صورت نام فایل
        $displayName = !empty($file['title']) ? $file['title'] : $file['file_name'];
        
        // محدود کردن طول نام
        if (mb_strlen($displayName) > 30) {
            $displayName = mb_substr($displayName, 0, 27) . '...';
        }
        
        $buttons[] = [
            ['text' => "📄 {$counter}. {$displayName}", 'callback_data' => 'view_downloaded_' . $file['id']]
        ];
        
        $counter++;
        
        // محدودیت تعداد دکمه‌ها (حداکثر 20 فایل در هر پیام)
        if ($counter > 20) {
            break;
        }
    }
    
    $keyboard = Telegram::makeInlineKeyboard($buttons);
    
    Telegram::sendMessage($chatId, $message, $keyboard);
}

/**
 * نمایش پیشنمایش فایل دانلود شده
 */
function handleViewDownloadedFile($chatId, $userId, $fileId, $messageId) {
    $file = Database::getFile($fileId);
    
    if (!$file) {
        Telegram::request('answerCallbackQuery', [
            'callback_query_id' => $callbackQuery['id'],
            'text' => '❌ فایل یافت نشد!',
            'show_alert' => true
        ]);
        return;
    }
    
    // بررسی اینکه کاربر این فایل را دانلود کرده است
    $downloadedFiles = Database::getUserDownloadedFiles($userId);
    $hasDownloaded = false;
    
    foreach ($downloadedFiles as $df) {
        if ($df['id'] == $fileId) {
            $hasDownloaded = true;
            break;
        }
    }
    
    if (!$hasDownloaded) {
        Telegram::request('answerCallbackQuery', [
            'callback_query_id' => $callbackQuery['id'],
            'text' => '❌ شما این فایل را دانلود نکرده‌اید!',
            'show_alert' => true
        ]);
        return;
    }
    
    // ساخت پیشنمایش
    $previewMessage = "📦 " . HTMLParser::format("پیشنمایش فایل", true) . "\n\n";
    
    $fileInfo = "";
    
    // نمایش تایتل اگر وجود دارد
    if (!empty($file['title'])) {
        $fileInfo .= "📌 " . HTMLParser::format("عنوان: ", true) . $file['title'] . "\n";
    }
    
    $fileInfo .= "📄 نام فایل: " . HTMLParser::format($file['file_name'], true) . "\n";
    $fileInfo .= "💾 حجم: " . HTMLParser::format(HTMLParser::formatFileSize($file['file_size']), true) . "\n";
    $fileInfo .= "📅 تاریخ آپلود: " . HTMLParser::format(date('Y/m/d', strtotime($file['upload_date'])), true) . "\n";
    
    // نمایش کپشن اگر وجود دارد
    if (!empty($file['caption'])) {
        $fileInfo .= "\n📝 توضیحات:\n" . $file['caption'];
    }
    
    // نمایش امتیاز
    if (isset($file['total_ratings']) && $file['total_ratings'] > 0) {
        $fileInfo .= "\n\n⭐ امتیاز: " . HTMLParser::ratingDisplay($file['average_rating'], $file['total_ratings']);
    }
    
    $previewMessage .= HTMLParser::blockquote($fileInfo);
    $previewMessage .= "\n\nآیا مایل به دانلود مجدد این فایل هستید؟";
    
    // دکمه‌های عملیات
    $keyboard = Telegram::makeInlineKeyboard([
        [
            ['text' => '✅ دانلود مجدد', 'callback_data' => 'redownload_file_' . $fileId]
        ],
        [
            ['text' => '🔙 بازگشت', 'callback_data' => 'back_to_downloads']
        ]
    ]);
    
    Telegram::request('editMessageText', [
        'chat_id' => $chatId,
        'message_id' => $messageId,
        'text' => $previewMessage,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode($keyboard)
    ]);
}

/**
 * دانلود مجدد فایل
 */
function handleRedownloadFile($chatId, $userId, $fileId) {
    $file = Database::getFile($fileId);
    
    if (!$file) {
        Telegram::sendMessage($chatId, "❌ فایل یافت نشد!");
        return;
    }
    
    // بررسی فعال بودن فایل
    if (!$file['is_active']) {
        Telegram::sendMessage($chatId, "❌ این فایل دیگر فعال نیست!");
        return;
    }
    
    // بررسی تاریخ انقضا
    if (!empty($file['expire_date'])) {
        $expireTime = strtotime($file['expire_date']);
        if ($expireTime && time() > $expireTime) {
            Telegram::sendMessage($chatId, "⏰ این فایل منقضی شده است!");
            return;
        }
    }
    
    // ارسال فایل
    Telegram::sendChatAction($chatId, 'upload_document');
    
    $fileType = $file['file_type'];
    $fileId = $file['file_id'];
    $caption = !empty($file['caption']) ? $file['caption'] : ("📁 " . $file['file_name']);
    
    if (in_array($fileType, ['photo', 'image'])) {
        Telegram::sendPhoto($chatId, $fileId, $caption);
    } elseif ($fileType == 'video') {
        Telegram::sendVideo($chatId, $fileId, $caption);
    } elseif ($fileType == 'audio') {
        Telegram::sendAudio($chatId, $fileId, $caption);
    } elseif ($fileType == 'voice') {
        Telegram::sendVoice($chatId, $fileId, $caption);
    } else {
        Telegram::sendDocument($chatId, $fileId, $caption);
    }
    
    // پیام تشکر
    $thanksMessage = "✅ " . HTMLParser::format("فایل با موفقیت ارسال شد", true) . "\n\n";
    $thanksMessage .= HTMLParser::blockquote("از استفاده شما متشکریم! 💙");
    
    Telegram::sendMessage($chatId, $thanksMessage, Telegram::getUserKeyboard());
}

/**
 * نمایش لیست بهترین فایل‌ها
 */
function handleTopRatedFiles($chatId) {
    $files = Database::getTopRatedFiles(10);
    
    if (empty($files)) {
        $message = "⭐ " . HTMLParser::format("بهترین فایل‌ها", true) . "\n\n";
        $message .= HTMLParser::blockquote("هنوز فایلی امتیاز نگرفته است.");
        
        Telegram::sendMessage($chatId, $message, Telegram::getUserKeyboard());
        return;
    }
    
    $message = "⭐ " . HTMLParser::format("بهترین فایل‌ها", true) . "\n\n";
    $message .= HTMLParser::blockquote("Top " . count($files) . " فایل بر اساس امتیاز");
    $message .= "\n\n";
    
    $counter = 1;
    foreach ($files as $file) {
        $medal = '';
        if ($counter == 1) $medal = '🥇';
        elseif ($counter == 2) $medal = '🥈';
        elseif ($counter == 3) $medal = '🥉';
        else $medal = "{$counter}.";
        
        $displayName = !empty($file['title']) ? $file['title'] : $file['file_name'];
        if (mb_strlen($displayName) > 25) {
            $displayName = mb_substr($displayName, 0, 22) . '...';
        }
        
        $rating = HTMLParser::ratingDisplay($file['average_rating'], $file['total_ratings']);
        
        $message .= "{$medal} {$displayName}\n";
        $message .= "   {$rating}\n";
        $message .= "   📥 {$file['download_count']} دانلود\n\n";
        
        $counter++;
    }
    
    Telegram::sendMessage($chatId, $message, Telegram::getUserKeyboard());
}

/**
 * درخواست امتیازدهی بعد از دانلود
 */
function askForRating($chatId, $userId, $fileId) {
    $file = Database::getFile($fileId);
    
    if (!$file) {
        return;
    }
    
    // بررسی قبلاً امتیاز داده یا نه
    $existingRating = Database::getUserRatingForFile($fileId, $userId);
    
    $message = "⭐ " . HTMLParser::format("امتیازدهی به فایل", true) . "\n\n";
    
    if ($existingRating) {
        $message .= HTMLParser::blockquote("شما قبلاً به این فایل {$existingRating} ستاره داده‌اید");
        $message .= "\n\nآیا می‌خواهید امتیاز خود را تغییر دهید؟";
    } else {
        $message .= HTMLParser::blockquote("نظر شما درباره این فایل چیست؟");
        $message .= "\n\nلطفاً امتیاز خود را انتخاب کنید:";
    }
    
    $keyboard = Telegram::makeInlineKeyboard([
        [
            ['text' => '⭐⭐⭐⭐⭐ (5)', 'callback_data' => 'rate_' . $fileId . '_5'],
        ],
        [
            ['text' => '⭐⭐⭐⭐☆ (4)', 'callback_data' => 'rate_' . $fileId . '_4'],
        ],
        [
            ['text' => '⭐⭐⭐☆☆ (3)', 'callback_data' => 'rate_' . $fileId . '_3'],
        ],
        [
            ['text' => '⭐⭐☆☆☆ (2)', 'callback_data' => 'rate_' . $fileId . '_2'],
        ],
        [
            ['text' => '⭐☆☆☆☆ (1)', 'callback_data' => 'rate_' . $fileId . '_1'],
        ],
        [
            ['text' => '❌ بعداً', 'callback_data' => 'skip_rating']
        ]
    ]);
    
    Telegram::sendMessage($chatId, $message, $keyboard);
}

/**
 * ثبت امتیاز
 */
function handleRating($chatId, $userId, $fileId, $rating, $messageId) {
    $result = Database::rateFile($fileId, $userId, $rating);
    
    if ($result) {
        $file = Database::getFile($fileId);
        
        $message = "✅ " . HTMLParser::format("متشکریم!", true) . "\n\n";
        $message .= "امتیاز شما: " . HTMLParser::stars($rating) . " ({$rating}/5)\n\n";
        
        if ($file && isset($file['average_rating'])) {
            $message .= "میانگین فایل: " . HTMLParser::ratingDisplay($file['average_rating'], $file['total_ratings']);
        }
        
        Telegram::request('editMessageText', [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $message,
            'parse_mode' => 'HTML'
        ]);
    } else {
        Telegram::request('answerCallbackQuery', [
            'callback_query_id' => $callbackQuery['id'],
            'text' => '❌ خطا در ثبت امتیاز!',
            'show_alert' => true
        ]);
    }
}
