<?php
/**
 * هندلر پنل ادمین
 * Admin Panel Handler
 */

/**
 * نمایش پنل ادمین
 */
function showAdminPanel($chatId) {
    $message = "🎛 " . HTMLParser::format("پنل مدیریت", true) . "\n\n";
    $message .= HTMLParser::blockquote("به پنل مدیریت ربات خوش آمدید");
    $message .= "\n\nاز دکمه‌های زیر برای مدیریت ربات استفاده کنید.";
    
    Telegram::sendMessage($chatId, $message, Telegram::getAdminKeyboard());
}

/**
 * نمایش آمار کاربران
 */
function handleStats($chatId) {
    $stats = Database::getStats();
    
    $message = "📊 " . HTMLParser::format("آمار کلی ربات", true) . "\n\n";
    
    $message .= "👥 تعداد کاربران: " . HTMLParser::format(number_format($stats['total_users']), true) . "\n";
    $message .= "📁 تعداد فایل‌ها: " . HTMLParser::format(number_format($stats['total_files']), true) . "\n";
    $message .= "📥 کل دانلودها: " . HTMLParser::format(number_format($stats['total_downloads']), true) . "\n";
    $message .= "💾 حجم کل فایل‌ها: " . HTMLParser::format(HTMLParser::formatFileSize($stats['total_size']), true) . "\n\n";
    
    $message .= HTMLParser::blockquote("آمار به‌روز شده در " . date('Y/m/d H:i'));
    
    // دکمه جستجوی کاربر
    $keyboard = Telegram::makeKeyboard([
        [
            ['text' => '🔍 جستجوی کاربر']
        ],
        [
            ['text' => '🔙 بازگشت']
        ]
    ]);
    
    Telegram::sendMessage($chatId, $message, $keyboard);
}

/**
 * راهنمای جستجوی کاربر
 */
function handleSearchUserGuide($chatId) {
    $message = "🔍 " . HTMLParser::format("جستجوی کاربر", true) . "\n\n";
    $message .= HTMLParser::blockquote("برای جستجوی کاربر، ID عددی یا username را ارسال کنید");
    $message .= "\n\n📝 مثال:\n";
    $message .= HTMLParser::format("123456789", false, false, true) . "\n";
    $message .= HTMLParser::format("@username", false, false, true) . "\n\n";
    $message .= "ID یا username کاربر را ارسال کنید:";
    
    // تنظیم state برای دریافت query
    Database::setUploadState($chatId, 'awaiting_user_search', []);
    
    Telegram::sendMessage($chatId, $message);
}

/**
 * جستجو و نمایش اطلاعات کاربر
 */
function handleUserSearch($chatId, $query) {
    $user = Database::searchUser($query);
    
    if (!$user) {
        $message = "❌ " . HTMLParser::format("کاربر یافت نشد!", true) . "\n\n";
        $message .= HTMLParser::blockquote("هیچ کاربری با این مشخصات در ربات ثبت نشده است.");
        
        Telegram::sendMessage($chatId, $message);
        Database::clearUploadState($chatId);
        return;
    }
    
    // دریافت آمار دانلود کاربر
    $stats = Database::getUserDownloadStats($user['user_id']);
    
    // ساخت پیام اطلاعات
    $message = "👤 " . HTMLParser::format("اطلاعات کاربر", true) . "\n\n";
    
    $userInfo = "🆔 " . HTMLParser::format("ID: ", true) . $user['user_id'] . "\n";
    $userInfo .= "👤 " . HTMLParser::format("نام: ", true) . ($user['first_name'] ?: 'ندارد');
    if ($user['last_name']) {
        $userInfo .= " " . $user['last_name'];
    }
    $userInfo .= "\n";
    
    if ($user['username']) {
        $userInfo .= "📱 " . HTMLParser::format("یوزرنیم: ", true) . "@" . $user['username'] . "\n";
    }
    
    $userInfo .= "📅 " . HTMLParser::format("تاریخ عضویت: ", true) . date('Y/m/d', strtotime($user['join_date'])) . "\n";
    $userInfo .= "📥 " . HTMLParser::format("تعداد دانلود: ", true) . number_format($stats['download_count']) . "\n";
    $userInfo .= "💾 " . HTMLParser::format("حجم دانلود: ", true) . HTMLParser::formatFileSize($stats['total_size']) . "\n";
    
    $blockStatus = $user['is_blocked'] ? "🔴 بلاک شده" : "🟢 فعال";
    $userInfo .= "\n📊 " . HTMLParser::format("وضعیت: ", true) . $blockStatus;
    
    $message .= HTMLParser::blockquote($userInfo);
    
    // دکمه‌های مدیریت
    $buttons = [];
    
    if ($user['is_blocked']) {
        $buttons[] = [
            ['text' => '✅ آنبلاک کاربر', 'callback_data' => 'unblock_' . $user['user_id']]
        ];
    } else {
        $buttons[] = [
            ['text' => '🚫 بلاک کاربر', 'callback_data' => 'block_' . $user['user_id']]
        ];
    }
    
    $keyboard = Telegram::makeInlineKeyboard($buttons);
    
    Telegram::sendMessage($chatId, $message, $keyboard);
    Database::clearUploadState($chatId);
}

/**
 * بلاک کردن کاربر
 */
function handleBlockUser($chatId, $userId, $messageId) {
    $result = Database::blockUser($userId);
    
    if ($result) {
        $user = Database::getUser($userId);
        
        $message = "🚫 " . HTMLParser::format("کاربر با موفقیت بلاک شد!", true) . "\n\n";
        
        $userInfo = "👤 " . ($user['first_name'] ?: 'کاربر');
        if ($user['username']) {
            $userInfo .= " (@" . $user['username'] . ")";
        }
        $userInfo .= "\n🆔 " . $userId;
        
        $message .= HTMLParser::blockquote($userInfo);
        $message .= "\n\nاین کاربر دیگر نمی‌تواند از ربات استفاده کند.";
        
        // دکمه آنبلاک
        $keyboard = Telegram::makeInlineKeyboard([
            [
                ['text' => '✅ آنبلاک کاربر', 'callback_data' => 'unblock_' . $userId]
            ]
        ]);
        
        Telegram::request('editMessageText', [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $message,
            'parse_mode' => 'HTML',
            'reply_markup' => json_encode($keyboard)
        ]);
    } else {
        Telegram::request('answerCallbackQuery', [
            'callback_query_id' => $callbackQuery['id'],
            'text' => '❌ خطا در بلاک کاربر!',
            'show_alert' => true
        ]);
    }
}

/**
 * آنبلاک کردن کاربر
 */
function handleUnblockUser($chatId, $userId, $messageId) {
    $result = Database::unblockUser($userId);
    
    if ($result) {
        $user = Database::getUser($userId);
        
        $message = "✅ " . HTMLParser::format("کاربر با موفقیت آنبلاک شد!", true) . "\n\n";
        
        $userInfo = "👤 " . ($user['first_name'] ?: 'کاربر');
        if ($user['username']) {
            $userInfo .= " (@" . $user['username'] . ")";
        }
        $userInfo .= "\n🆔 " . $userId;
        
        $message .= HTMLParser::blockquote($userInfo);
        $message .= "\n\nاین کاربر می‌تواند دوباره از ربات استفاده کند.";
        
        // دکمه بلاک
        $keyboard = Telegram::makeInlineKeyboard([
            [
                ['text' => '🚫 بلاک کاربر', 'callback_data' => 'block_' . $userId]
            ]
        ]);
        
        Telegram::request('editMessageText', [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $message,
            'parse_mode' => 'HTML',
            'reply_markup' => json_encode($keyboard)
        ]);
    } else {
        Telegram::request('answerCallbackQuery', [
            'callback_query_id' => $callbackQuery['id'],
            'text' => '❌ خطا در آنبلاک کاربر!',
            'show_alert' => true
        ]);
    }
}

/**
 * نمایش وضعیت سرور
 */
function handleServerStatus($chatId) {
    $message = "🖥 " . HTMLParser::format("وضعیت سرور", true) . "\n\n";
    
    // استفاده از حافظه
    $memoryUsage = memory_get_usage(true);
    $memoryLimit = ini_get('memory_limit');
    $message .= "🧠 حافظه استفاده شده: " . HTMLParser::format(HTMLParser::formatFileSize($memoryUsage)) . "\n";
    $message .= "📌 محدودیت حافظه: " . HTMLParser::format($memoryLimit) . "\n\n";
    
    // فضای دیسک
    $diskFree = @disk_free_space(BASE_PATH);
    $diskTotal = @disk_total_space(BASE_PATH);
    
    if ($diskFree && $diskTotal) {
        $diskUsed = $diskTotal - $diskFree;
        $diskPercent = round(($diskUsed / $diskTotal) * 100, 2);
        
        $message .= "💿 فضای دیسک:\n";
        $message .= "   استفاده شده: " . HTMLParser::format(HTMLParser::formatFileSize($diskUsed)) . "\n";
        $message .= "   کل: " . HTMLParser::format(HTMLParser::formatFileSize($diskTotal)) . "\n";
        $message .= "   درصد استفاده: " . HTMLParser::format($diskPercent . '%', true) . "\n\n";
    }
    
    // نسخه PHP
    $message .= "🐘 نسخه PHP: " . HTMLParser::format(PHP_VERSION) . "\n";
    
    // زمان سرور
    $message .= "🕐 زمان سرور: " . HTMLParser::format(date('Y/m/d H:i:s')) . "\n\n";
    
    // آمار فایل‌های ربات از دیتابیس
    $files = Database::getAllFiles();
    $totalFiles = count($files);
    $totalSize = 0;
    
    foreach ($files as $file) {
        $totalSize += $file['file_size'] ?? 0;
    }
    
    $message .= "📊 " . HTMLParser::format("آمار فایل‌های ربات:", true) . "\n";
    $message .= "📁 تعداد فایل‌ها: " . HTMLParser::format(number_format($totalFiles), true) . "\n";
    $message .= "💾 حجم کل: " . HTMLParser::format(HTMLParser::formatFileSize($totalSize), true) . "\n\n";
    
    // تعداد فایل‌های فیزیکی در پوشه uploads
    $physicalFiles = @glob(UPLOADS_PATH . '*');
    $physicalCount = is_array($physicalFiles) ? count($physicalFiles) : 0;
    $message .= "📦 فایل‌های محلی: " . HTMLParser::format(number_format($physicalCount), true) . "\n";
    
    $message .= "\n" . HTMLParser::blockquote("✅ سرور به درستی کار می‌کند");
    
    Telegram::sendMessage($chatId, $message, Telegram::getAdminKeyboard());
}

/**
 * نمایش منوی تنظیمات
 */
function handleSettings($chatId) {
    $settings = Database::getSettings();
    
    $message = "⚙️ " . HTMLParser::format("تنظیمات ربات", true) . "\n\n";
    
    // وضعیت تعمیر و نگهداری
    $maintenanceStatus = $settings['maintenance_mode'] ? '🔴 فعال' : '🟢 غیرفعال';
    $message .= "🔧 حالت تعمیر: " . HTMLParser::format($maintenanceStatus, true) . "\n";
    
    // وضعیت جوین اجباری
    $forceJoinStatus = $settings['force_join'] ? '� فعال' : '🔴 غیرفعال';
    $message .= "🔗 جوین اجباری: " . HTMLParser::format($forceJoinStatus, true) . "\n";
    
    // تعداد کانال‌ها
    $channels = Database::getForceJoinChannels();
    $channelCount = count($channels);
    $message .= "📢 تعداد کانال‌ها: " . HTMLParser::format($channelCount, true) . "\n\n";
    
    $message .= HTMLParser::blockquote("از دکمه‌های زیر برای تغییر تنظیمات استفاده کنید");
    
    Telegram::sendMessage($chatId, $message, Telegram::getSettingsKeyboard());
}

/**
 * مدیریت کانال‌های جوین اجباری
 */
function handleChannelsMenu($chatId) {
    $channels = Database::getForceJoinChannels();
    
    $message = "📢 " . HTMLParser::format("مدیریت کانال‌های جوین اجباری", true) . "\n\n";
    
    if (empty($channels)) {
        $message .= HTMLParser::blockquote("هیچ کانالی ثبت نشده است");
        $message .= "\n\nبرای افزودن کانال جدید، از دکمه زیر استفاده کنید.";
    } else {
        $message .= "تعداد کانال‌ها: " . HTMLParser::format(count($channels), true) . "\n\n";
        $message .= HTMLParser::blockquote("از دکمه‌های زیر برای مدیریت کانال‌ها استفاده کنید");
    }
    
    Telegram::sendMessage($chatId, $message, Telegram::getChannelsKeyboard());
}

/**
 * نمایش لیست کانال‌ها
 */
function handleChannelsList($chatId) {
    $channels = Database::getForceJoinChannels();
    
    if (empty($channels)) {
        $message = "📋 " . HTMLParser::format("لیست کانال‌ها", true) . "\n\n";
        $message .= HTMLParser::blockquote("هیچ کانالی ثبت نشده است");
        $message .= "\n\n💡 برای افزودن کانال از دکمه " . HTMLParser::format("➕ افزودن کانال", true) . " استفاده کنید.";
        
        Telegram::sendMessage($chatId, $message, Telegram::getChannelsKeyboard());
        return;
    }
    
    $message = "📋 " . HTMLParser::format("لیست کانال‌های جوین اجباری", true) . "\n";
    $message .= "تعداد: " . HTMLParser::format(count($channels), true) . "\n\n";
    
    $counter = 1;
    foreach ($channels as $channel) {
        $channelText = "{$counter}. " . ($channel['title'] ?: 'کانال') . "\n";
        $channelText .= "   🆔 " . $channel['username'] . "\n";
        $channelText .= "   🔑 ID: " . $channel['id'];
        
        $message .= HTMLParser::blockquote($channelText) . "\n\n";
        $counter++;
    }
    
    $message .= "💡 " . HTMLParser::format("نکته:", true) . " برای حذف کانال از دکمه " . HTMLParser::format("➖ حذف کانال", true) . " استفاده کنید.";
    
    Telegram::sendMessage($chatId, $message, Telegram::getChannelsKeyboard());
}

/**
 * راهنمای افزودن کانال
 */
function handleAddChannelGuide($chatId) {
    $message = "➕ " . HTMLParser::format("افزودن کانال جدید", true) . "\n\n";
    
    $message .= HTMLParser::blockquote("مراحل افزودن کانال") . "\n\n";
    
    $message .= "1️⃣ ربات را در کانال خود " . HTMLParser::format("ادمین", true) . " کنید\n";
    $message .= "2️⃣ از دستور زیر استفاده کنید:\n\n";
    
    $message .= HTMLParser::format("/addchannel @username", false, false, true) . "\n\n";
    
    $message .= "📝 " . HTMLParser::format("مثال:", true) . "\n";
    $message .= HTMLParser::format("/addchannel @mychannel", false, false, true) . "\n\n";
    
    $message .= HTMLParser::blockquote("⚠️ ربات باید در کانال ادمین باشد");
    
    Telegram::sendMessage($chatId, $message, Telegram::getChannelsKeyboard());
}

/**
 * افزودن کانال
 */
function handleAddChannel($chatId, $channelUsername) {
    // حذف @ اگر وجود داشت
    $channelUsername = ltrim($channelUsername, '@');
    $channelId = '@' . $channelUsername;
    
    // بررسی اینکه ربات در کانال ادمین است
    $chatInfo = Telegram::request('getChat', ['chat_id' => $channelId]);
    
    if (!isset($chatInfo['ok']) || !$chatInfo['ok']) {
        $message = "❌ " . HTMLParser::format("خطا!", true) . "\n\n";
        $message .= HTMLParser::blockquote("کانال یافت نشد یا ربات در آن عضو نیست");
        $message .= "\n\n💡 ابتدا ربات را در کانال ادمین کنید.";
        
        Telegram::sendMessage($chatId, $message, Telegram::getChannelsKeyboard());
        return;
    }
    
    $channelTitle = $chatInfo['result']['title'] ?? 'کانال';
    
    // افزودن کانال
    $result = Database::addForceJoinChannel('@' . $channelUsername, $channelId, $channelTitle);
    
    if ($result) {
        $message = "✅ " . HTMLParser::format("کانال با موفقیت اضافه شد!", true) . "\n\n";
        
        $channelInfo = "📢 " . $channelTitle . "\n";
        $channelInfo .= "🆔 @" . $channelUsername;
        
        $message .= HTMLParser::blockquote($channelInfo) . "\n\n";
        
        $message .= "اکنون کاربران باید عضو این کانال باشند.";
    } else {
        $message = "⚠️ " . HTMLParser::format("این کانال قبلاً اضافه شده است!", true);
    }
    
    Telegram::sendMessage($chatId, $message, Telegram::getChannelsKeyboard());
}

/**
 * راهنمای حذف کانال
 */
function handleRemoveChannelGuide($chatId) {
    $channels = Database::getForceJoinChannels();
    
    if (empty($channels)) {
        $message = "❌ " . HTMLParser::format("هیچ کانالی برای حذف وجود ندارد!", true);
        Telegram::sendMessage($chatId, $message, Telegram::getChannelsKeyboard());
        return;
    }
    
    $message = "➖ " . HTMLParser::format("حذف کانال", true) . "\n\n";
    
    $message .= HTMLParser::blockquote("لیست کانال‌های ثبت شده") . "\n\n";
    
    $counter = 1;
    foreach ($channels as $channel) {
        $message .= "{$counter}. " . HTMLParser::format($channel['title'] ?: 'کانال', true) . "\n";
        $message .= "   🆔 " . HTMLParser::format($channel['username']) . "\n";
        $message .= "   🔑 ID: " . HTMLParser::format($channel['id'], false, false, true) . "\n\n";
        $counter++;
    }
    
    $message .= "📝 برای حذف از دستور زیر استفاده کنید:\n\n";
    $message .= HTMLParser::format("/removechannel @username", false, false, true) . "\n\n";
    
    $message .= "مثال:\n";
    $message .= HTMLParser::format("/removechannel @mychannel", false, false, true);
    
    Telegram::sendMessage($chatId, $message, Telegram::getChannelsKeyboard());
}

/**
 * حذف کانال
 */
function handleRemoveChannel($chatId, $channelUsername) {
    // حذف @ اگر وجود داشت
    $channelUsername = ltrim($channelUsername, '@');
    $channelId = '@' . $channelUsername;
    
    $result = Database::removeForceJoinChannel($channelId);
    
    if ($result) {
        $message = "✅ " . HTMLParser::format("کانال با موفقیت حذف شد!", true) . "\n\n";
        $message .= HTMLParser::blockquote("🆔 @" . $channelUsername);
    } else {
        $message = "❌ " . HTMLParser::format("کانال یافت نشد!", true) . "\n\n";
        $message .= "این کانال در لیست جوین اجباری ثبت نشده است.";
    }
    
    Telegram::sendMessage($chatId, $message, Telegram::getChannelsKeyboard());
}

/**
 * نمایش منوی اطلاع‌رسانی
 */
function handleNotificationMenu($chatId) {
    $settings = Database::getSettings();
    
    $message = "🔔 " . HTMLParser::format("تنظیمات اطلاع‌رسانی آپلود", true) . "\n\n";
    
    $statusText = $settings['notification_enabled'] ? '✅ فعال' : '❌ غیرفعال';
    $message .= "وضعیت: " . HTMLParser::format($statusText, true) . "\n\n";
    
    if ($settings['notification_enabled']) {
        $toUsersText = $settings['notification_to_users'] ? '✅' : '❌';
        $toChannelText = $settings['notification_to_channel'] ? '✅' : '❌';
        
        $message .= "ارسال به کاربران: " . $toUsersText . "\n";
        $message .= "ارسال به کانال: " . $toChannelText . "\n";
        
        if (!empty($settings['notification_channel'])) {
            $message .= "کانال: " . HTMLParser::format($settings['notification_channel'], false, false, true) . "\n";
        } else {
            $message .= "کانال: " . HTMLParser::format("تنظیم نشده", true) . "\n";
        }
    }
    
    $message .= "\n" . HTMLParser::blockquote("از منوی زیر تنظیمات را مدیریت کنید");
    
    $keyboard = Telegram::makeKeyboard([
        [
            ['text' => $settings['notification_enabled'] ? '🔴 غیرفعال کردن' : '🟢 فعال کردن']
        ],
        [
            ['text' => '👥 ارسال به کاربران: ' . ($settings['notification_to_users'] ? '✅' : '❌')]
        ],
        [
            ['text' => '📢 ارسال به کانال: ' . ($settings['notification_to_channel'] ? '✅' : '❌')]
        ],
        [
            ['text' => '⚙️ تنظیم کانال']
        ],
        [
            ['text' => '🔙 بازگشت به تنظیمات']
        ]
    ]);
    
    Telegram::sendMessage($chatId, $message, $keyboard);
}

/**
 * تغییر وضعیت اطلاع‌رسانی
 */
function toggleNotification($chatId) {
    $settings = Database::getSettings();
    $newStatus = !$settings['notification_enabled'];
    
    Database::updateSetting('notification_enabled', $newStatus);
    
    $statusText = $newStatus ? 'فعال' : 'غیرفعال';
    $emoji = $newStatus ? '🟢' : '🔴';
    
    $message = "{$emoji} اطلاع‌رسانی آپلود " . HTMLParser::format($statusText, true) . " شد!\n\n";
    
    if ($newStatus) {
        $message .= HTMLParser::blockquote("هنگام آپلود فایل جدید، به کاربران و/یا کانال اطلاع داده می‌شود");
    } else {
        $message .= HTMLParser::blockquote("اطلاع‌رسانی خودکار غیرفعال است");
    }
    
    handleNotificationMenu($chatId);
}

/**
 * تغییر وضعیت ارسال به کاربران
 */
function toggleNotificationToUsers($chatId) {
    $settings = Database::getSettings();
    $newStatus = !$settings['notification_to_users'];
    
    Database::updateSetting('notification_to_users', $newStatus);
    
    $statusText = $newStatus ? '✅ فعال' : '❌ غیرفعال';
    Telegram::sendMessage($chatId, "ارسال به کاربران: {$statusText}");
    
    handleNotificationMenu($chatId);
}

/**
 * تغییر وضعیت ارسال به کانال
 */
function toggleNotificationToChannel($chatId) {
    $settings = Database::getSettings();
    $newStatus = !$settings['notification_to_channel'];
    
    Database::updateSetting('notification_to_channel', $newStatus);
    
    $statusText = $newStatus ? '✅ فعال' : '❌ غیرفعال';
    Telegram::sendMessage($chatId, "ارسال به کانال: {$statusText}");
    
    handleNotificationMenu($chatId);
}

/**
 * راهنمای تنظیم کانال اطلاع‌رسانی
 */
function handleSetNotificationChannelGuide($chatId) {
    $message = "⚙️ " . HTMLParser::format("تنظیم کانال اطلاع‌رسانی", true) . "\n\n";
    $message .= HTMLParser::blockquote("یوزرنیم یا ID کانال را ارسال کنید");
    $message .= "\n\n📝 مثال:\n";
    $message .= HTMLParser::format("@mychannel", false, false, true) . "\n";
    $message .= "یا\n";
    $message .= HTMLParser::format("-1001234567890", false, false, true) . "\n\n";
    $message .= "⚠️ توجه: ربات باید در کانال ادمین باشد!";
    
    Database::setUploadState($chatId, 'awaiting_notification_channel', []);
    
    Telegram::sendMessage($chatId, $message);
}

/**
 * تنظیم کانال اطلاع‌رسانی
 */
function handleSetNotificationChannel($chatId, $channel) {
    // تمیز کردن ورودی
    $channel = trim($channel);
    
    if (strpos($channel, '@') !== 0 && !is_numeric($channel)) {
        $channel = '@' . $channel;
    }
    
    Database::updateSetting('notification_channel', $channel);
    Database::clearUploadState($chatId);
    
    $message = "✅ " . HTMLParser::format("کانال با موفقیت تنظیم شد!", true) . "\n\n";
    $message .= HTMLParser::blockquote("کانال: " . $channel);
    $message .= "\n\n⚠️ اطمینان حاصل کنید که ربات در کانال ادمین است.";
    
    Telegram::sendMessage($chatId, $message);
    handleNotificationMenu($chatId);
}

/**
 * ارسال اطلاع‌رسانی آپلود جدید
 */
function sendNewFileNotification($fileData, $uniqueId) {
    $settings = Database::getSettings();
    
    // بررسی فعال بودن
    if (!$settings['notification_enabled']) {
        return;
    }
    
    // اگر هیچکدام فعال نباشد
    if (!$settings['notification_to_users'] && !$settings['notification_to_channel']) {
        return;
    }
    
    // ساخت پیام اطلاع‌رسانی
    $botUsername = BOT_USERNAME;
    $downloadLink = "https://t.me/{$botUsername}?start={$uniqueId}";
    
    $message = "🆕 " . HTMLParser::format("فایل جدید آپلود شد!", true) . "\n\n";
    
    if (!empty($fileData['title'])) {
        $message .= "🏷 عنوان: " . HTMLParser::format($fileData['title'], true) . "\n";
    }
    
    $message .= "📁 نام: " . HTMLParser::format($fileData['file_name'], false, false, true) . "\n";
    $message .= "💾 حجم: " . HTMLParser::format(HTMLParser::formatFileSize($fileData['file_size'])) . "\n";
    $message .= "📊 نوع: " . HTMLParser::format($fileData['file_type'], true) . "\n";
    
    if (!empty($fileData['caption'])) {
        $message .= "\n💬 توضیحات:\n";
        $message .= HTMLParser::blockquote($fileData['caption']) . "\n";
    }
    
    $message .= "\n🔗 لینک دانلود:\n";
    $message .= HTMLParser::format($downloadLink, false, false, true);
    
    // ارسال به کاربران
    if ($settings['notification_to_users']) {
        $users = Database::read(USERS_DB);
        $sentCount = 0;
        
        foreach ($users as $user) {
            // عدم ارسال به کاربران بلاک شده
            if (!empty($user['is_blocked'])) {
                continue;
            }
            
            $result = Telegram::sendMessage($user['user_id'], $message);
            
            if (isset($result['ok']) && $result['ok']) {
                $sentCount++;
            }
            
            // جلوگیری از flood
            usleep(50000); // 50ms
        }
    }
    
    // ارسال به کانال
    if ($settings['notification_to_channel'] && !empty($settings['notification_channel'])) {
        Telegram::sendMessage($settings['notification_channel'], $message);
    }
}

/**
 * نمایش منوی مدیریت دکمه‌های کاربران
 */
function handleUserButtonsMenu($chatId) {
    $settings = Database::getSettings();
    
    $message = "🎛 " . HTMLParser::format("مدیریت دکمه‌های کاربران", true) . "\n\n";
    $message .= HTMLParser::blockquote("فعال/غیرفعال کردن دکمه‌های پنل کاربران");
    $message .= "\n\n";
    
    $infoStatus = (!isset($settings['user_button_info']) || $settings['user_button_info']) ? '✅' : '❌';
    $downloadsStatus = (!isset($settings['user_button_downloads']) || $settings['user_button_downloads']) ? '✅' : '❌';
    $topratedStatus = (!isset($settings['user_button_toprated']) || $settings['user_button_toprated']) ? '✅' : '❌';
    
    $message .= "👤 اطلاعات من: " . $infoStatus . "\n";
    $message .= "📥 فایل‌های دانلود شده: " . $downloadsStatus . "\n";
    $message .= "⭐ بهترین فایل‌ها: " . $topratedStatus . "\n";
    
    $keyboard = Telegram::makeKeyboard([
        [
            ['text' => '👤 اطلاعات من: ' . $infoStatus]
        ],
        [
            ['text' => '📥 فایل‌های دانلود شده: ' . $downloadsStatus]
        ],
        [
            ['text' => '⭐ بهترین فایل‌ها: ' . $topratedStatus]
        ],
        [
            ['text' => '🔙 بازگشت به تنظیمات']
        ]
    ]);
    
    Telegram::sendMessage($chatId, $message, $keyboard);
}

/**
 * تغییر وضعیت دکمه اطلاعات من
 */
function toggleUserButtonInfo($chatId) {
    $settings = Database::getSettings();
    $currentStatus = !isset($settings['user_button_info']) || $settings['user_button_info'];
    $newStatus = !$currentStatus;
    
    Database::updateSetting('user_button_info', $newStatus);
    
    $statusText = $newStatus ? '✅ فعال' : '❌ غیرفعال';
    Telegram::sendMessage($chatId, "دکمه 'اطلاعات من': {$statusText}");
    
    handleUserButtonsMenu($chatId);
}

/**
 * تغییر وضعیت دکمه فایل‌های دانلود شده
 */
function toggleUserButtonDownloads($chatId) {
    $settings = Database::getSettings();
    $currentStatus = !isset($settings['user_button_downloads']) || $settings['user_button_downloads'];
    $newStatus = !$currentStatus;
    
    Database::updateSetting('user_button_downloads', $newStatus);
    
    $statusText = $newStatus ? '✅ فعال' : '❌ غیرفعال';
    Telegram::sendMessage($chatId, "دکمه 'فایل‌های دانلود شده': {$statusText}");
    
    handleUserButtonsMenu($chatId);
}

/**
 * تغییر وضعیت دکمه بهترین فایل‌ها
 */
function toggleUserButtonTopRated($chatId) {
    $settings = Database::getSettings();
    $currentStatus = !isset($settings['user_button_toprated']) || $settings['user_button_toprated'];
    $newStatus = !$currentStatus;
    
    Database::updateSetting('user_button_toprated', $newStatus);
    
    $statusText = $newStatus ? '✅ فعال' : '❌ غیرفعال';
    Telegram::sendMessage($chatId, "دکمه 'بهترین فایل‌ها': {$statusText}");
    
    handleUserButtonsMenu($chatId);
}

/**
 * نمایش منوی اخطار به کاربران
 */
function handleWarningsMenu($chatId) {
    $message = "⚠️ " . HTMLParser::format("مدیریت اخطارها", true) . "\n\n";
    $message .= HTMLParser::blockquote("از این بخش می‌توانید به کاربران اخطار دهید");
    $message .= "\n\n💡 امکانات:\n";
    $message .= "• ارسال اخطار به کاربر\n";
    $message .= "• مشاهده اخطارهای کاربر\n";
    $message .= "• پاک کردن اخطارها\n";
    $message .= "• بلاک خودکار پس از 3 اخطار\n\n";
    $message .= "📝 برای ارسال اخطار:\n";
    $message .= HTMLParser::format("/warn <user_id> <دلیل>", false, false, true) . "\n\n";
    $message .= "📋 برای مشاهده اخطارها:\n";
    $message .= HTMLParser::format("/warnings <user_id>", false, false, true) . "\n\n";
    $message .= "🧹 برای پاک کردن اخطارها:\n";
    $message .= HTMLParser::format("/clearwarnings <user_id>", false, false, true);
    
    $keyboard = Telegram::makeKeyboard([
        [
            ['text' => '➕ اخطار جدید'],
            ['text' => '📋 مشاهده اخطارها']
        ],
        [
            ['text' => '🧹 پاک کردن اخطارها']
        ],
        [
            ['text' => '🔙 بازگشت']
        ]
    ]);
    
    Telegram::sendMessage($chatId, $message, $keyboard);
}

/**
 * راهنمای ارسال اخطار
 */
function handleSendWarningGuide($chatId, $userId) {
    $message = "➕ " . HTMLParser::format("ارسال اخطار جدید", true) . "\n\n";
    $message .= HTMLParser::blockquote("User ID و دلیل اخطار را به فرمت زیر ارسال کنید:");
    $message .= "\n\n" . HTMLParser::format("<user_id> <دلیل اخطار>", false, false, true) . "\n\n";
    $message .= "مثال:\n";
    $message .= HTMLParser::format("123456789 ارسال اسپم", false, false, true);
    
    Telegram::sendMessage($chatId, $message);
    Database::setUploadState($userId, 'awaiting_warning_info', []);
}

/**
 * ارسال اخطار به کاربر
 */
function handleSendWarning($chatId, $userId, $targetUserId, $reason) {
    if (!Telegram::isAdmin($userId)) {
        Telegram::sendMessage($chatId, "⛔ شما مجوز این عملیات را ندارید!");
        return;
    }
    
    if (!is_numeric($targetUserId)) {
        Telegram::sendMessage($chatId, "❌ User ID باید عدد باشد!");
        return;
    }
    
    $targetUserId = intval($targetUserId);
    
    // بررسی وجود کاربر
    $user = Database::getUserById($targetUserId);
    if (!$user) {
        Telegram::sendMessage($chatId, "❌ کاربر یافت نشد!");
        return;
    }
    
    // اضافه کردن اخطار
    Database::addWarning($targetUserId, $reason, $userId);
    $warningCount = Database::getUserWarningCount($targetUserId);
    
    // ارسال پیام به ادمین
    $userName = $user['first_name'];
    if (!empty($user['username'])) {
        $userName .= " (@" . $user['username'] . ")";
    }
    
    $adminMsg = "✅ اخطار با موفقیت ارسال شد!\n\n";
    $adminMsg .= "👤 کاربر: " . HTMLParser::format($userName, true) . "\n";
    $adminMsg .= "🆔 User ID: " . HTMLParser::format($targetUserId, false, false, true) . "\n";
    $adminMsg .= "⚠️ تعداد اخطارها: " . HTMLParser::format($warningCount, true) . "\n";
    $adminMsg .= "📝 دلیل: " . HTMLParser::format($reason, false, false, true);
    
    if ($warningCount >= 3) {
        $adminMsg .= "\n\n🚨 " . HTMLParser::format("هشدار!", true) . "\n";
        $adminMsg .= "این کاربر 3 اخطار دریافت کرده است.\nآیا می‌خواهید او را بلاک کنید؟";
        
        $keyboard = Telegram::makeInlineKeyboard([
            [
                ['text' => '🚫 بلاک کردن', 'callback_data' => 'block_user_' . $targetUserId],
                ['text' => '✅ نادیده گرفتن', 'callback_data' => 'ignore_warning']
            ]
        ]);
        
        Telegram::sendMessage($chatId, $adminMsg, $keyboard);
    } else {
        Telegram::sendMessage($chatId, $adminMsg, Telegram::getAdminKeyboard());
    }
    
    // ارسال پیام به کاربر
    $userMsg = "⚠️ " . HTMLParser::format("اخطار!", true) . "\n\n";
    $userMsg .= HTMLParser::blockquote("شما یک اخطار از مدیریت دریافت کردید");
    $userMsg .= "\n\n📝 دلیل: " . HTMLParser::format($reason, false, false, true) . "\n";
    $userMsg .= "⚠️ تعداد اخطارها: " . HTMLParser::format($warningCount . "/3", true) . "\n\n";
    
    if ($warningCount >= 3) {
        $userMsg .= "🚨 " . HTMLParser::format("توجه!", true) . "\n";
        $userMsg .= "شما به حداکثر تعداد اخطارها رسیده‌اید.\nممکن است حساب شما مسدود شود!";
    } else {
        $userMsg .= "💡 در صورت دریافت 3 اخطار، حساب شما مسدود خواهد شد.";
    }
    
    Telegram::sendMessage($targetUserId, $userMsg);
    
    Database::clearUploadState($userId);
}

/**
 * راهنمای مشاهده اخطارها
 */
function handleViewWarningsGuide($chatId, $userId) {
    $message = "📋 " . HTMLParser::format("مشاهده اخطارهای کاربر", true) . "\n\n";
    $message .= HTMLParser::blockquote("User ID کاربر را ارسال کنید:");
    $message .= "\n\nمثال:\n";
    $message .= HTMLParser::format("123456789", false, false, true);
    
    Telegram::sendMessage($chatId, $message);
    Database::setUploadState($userId, 'awaiting_view_warnings', []);
}

/**
 * مشاهده اخطارهای کاربر
 */
function handleViewWarnings($chatId, $targetUserId) {
    if (!is_numeric($targetUserId)) {
        Telegram::sendMessage($chatId, "❌ User ID باید عدد باشد!");
        return;
    }
    
    $targetUserId = intval($targetUserId);
    
    $user = Database::getUserById($targetUserId);
    if (!$user) {
        Telegram::sendMessage($chatId, "❌ کاربر یافت نشد!");
        return;
    }
    
    $warnings = Database::getUserWarnings($targetUserId);
    $warningCount = Database::getUserWarningCount($targetUserId);
    
    $userName = $user['first_name'];
    if (!empty($user['username'])) {
        $userName .= " (@" . $user['username'] . ")";
    }
    
    $message = "📋 " . HTMLParser::format("اخطارهای کاربر", true) . "\n\n";
    $message .= "👤 کاربر: " . HTMLParser::format($userName, true) . "\n";
    $message .= "🆔 User ID: " . HTMLParser::format($targetUserId, false, false, true) . "\n";
    $message .= "⚠️ تعداد اخطارها: " . HTMLParser::format($warningCount, true) . "\n\n";
    
    if (empty($warnings)) {
        $message .= HTMLParser::blockquote("این کاربر هیچ اخطاری ندارد");
    } else {
        $message .= "📝 لیست اخطارها:\n\n";
        
        foreach ($warnings as $index => $warning) {
            $message .= ($index + 1) . ". " . HTMLParser::format($warning['reason'], false, false, true) . "\n";
            $message .= "   📅 تاریخ: " . $warning['date'] . "\n";
            $message .= "   👮 ادمین: " . $warning['admin_id'] . "\n\n";
        }
        
        if ($warningCount >= 3) {
            $message .= "\n🚨 " . HTMLParser::format("این کاربر به حد بلاک رسیده است!", true);
        }
    }
    
    Telegram::sendMessage($chatId, $message, Telegram::getAdminKeyboard());
}

/**
 * راهنمای پاک کردن اخطارها
 */
function handleClearWarningsGuide($chatId, $userId) {
    $message = "🧹 " . HTMLParser::format("پاک کردن اخطارهای کاربر", true) . "\n\n";
    $message .= HTMLParser::blockquote("User ID کاربری که می‌خواهید اخطارهایش را پاک کنید ارسال کنید:");
    $message .= "\n\n⚠️ توجه: تمام اخطارهای کاربر پاک خواهد شد!\n\n";
    $message .= "مثال:\n";
    $message .= HTMLParser::format("123456789", false, false, true);
    
    Telegram::sendMessage($chatId, $message);
    Database::setUploadState($userId, 'awaiting_clear_warnings', []);
}

/**
 * پاک کردن اخطارهای کاربر
 */
function handleClearWarnings($chatId, $targetUserId) {
    if (!is_numeric($targetUserId)) {
        Telegram::sendMessage($chatId, "❌ User ID باید عدد باشد!");
        return;
    }
    
    $targetUserId = intval($targetUserId);
    
    $user = Database::getUserById($targetUserId);
    if (!$user) {
        Telegram::sendMessage($chatId, "❌ کاربر یافت نشد!");
        return;
    }
    
    $oldCount = Database::getUserWarningCount($targetUserId);
    
    if ($oldCount == 0) {
        Telegram::sendMessage($chatId, "ℹ️ این کاربر هیچ اخطاری ندارد!");
        return;
    }
    
    Database::clearUserWarnings($targetUserId);
    
    $userName = $user['first_name'];
    if (!empty($user['username'])) {
        $userName .= " (@" . $user['username'] . ")";
    }
    
    $message = "✅ اخطارها با موفقیت پاک شدند!\n\n";
    $message .= "👤 کاربر: " . HTMLParser::format($userName, true) . "\n";
    $message .= "🆔 User ID: " . HTMLParser::format($targetUserId, false, false, true) . "\n";
    $message .= "🗑 تعداد اخطارهای پاک شده: " . HTMLParser::format($oldCount, true);
    
    Telegram::sendMessage($chatId, $message, Telegram::getAdminKeyboard());
    
    // اطلاع به کاربر
    $userMsg = "✅ " . HTMLParser::format("خبر خوب!", true) . "\n\n";
    $userMsg .= "تمام اخطارهای شما توسط مدیریت " . HTMLParser::format("پاک شد", true) . ".\n\n";
    $userMsg .= "شما می‌توانید دوباره از ربات استفاده کنید.";
    
    Telegram::sendMessage($targetUserId, $userMsg);
}

/**
 * نمایش لیست اعتراضات
 */
function handleAppeals($chatId) {
    $appealsFile = __DIR__ . '/../data/appeals.json';
    $appeals = [];
    
    if (file_exists($appealsFile)) {
        $appeals = json_decode(file_get_contents($appealsFile), true) ?: [];
    }
    
    // فیلتر فقط اعتراضات در حال بررسی
    $pendingAppeals = array_filter($appeals, function($appeal) {
        return ($appeal['status'] ?? 'pending') === 'pending';
    });
    
    $message = "📋 " . HTMLParser::format("اعتراضات کاربران", true) . "\n\n";
    
    if (empty($pendingAppeals)) {
        $message .= HTMLParser::blockquote("هیچ اعتراض جدیدی وجود ندارد");
    } else {
        $message .= "تعداد اعتراضات در حال بررسی: " . HTMLParser::format(count($pendingAppeals), true) . "\n\n";
        
        foreach ($pendingAppeals as $index => $appeal) {
            $userName = $appeal['first_name'] ?? 'کاربر';
            if (!empty($appeal['username'])) {
                $userName .= " (@" . $appeal['username'] . ")";
            }
            
            $message .= ($index + 1) . ". " . HTMLParser::format($userName, true) . "\n";
            $message .= "   🆔 ID: " . ($appeal['user_id'] ?? 'نامشخص') . "\n";
            $message .= "   📝 متن: " . HTMLParser::format($appeal['appeal_text'] ?? '', false, false, true) . "\n";
            $message .= "   📅 تاریخ: " . ($appeal['date'] ?? 'نامشخص') . "\n\n";
        }
        
        $message .= "\n💡 برای مدیریت اعتراضات از پنل ادمین وب استفاده کنید.";
    }
    
    Telegram::sendMessage($chatId, $message, Telegram::getAdminKeyboard());
}

/**
 * نمایش منوی مدیریت ادمین‌ها
 */
function handleAdminsMenu($chatId, $userId) {
    // فقط ادمین اصلی می‌تواند ادمین‌ها را مدیریت کند
    if ($userId != ADMIN_ID) {
        Telegram::sendMessage($chatId, "⚠️ فقط ادمین اصلی می‌تواند ادمین‌ها را مدیریت کند!");
        return;
    }
    
    $adminsList = Database::getAdminsList();
    
    $message = "👥 " . HTMLParser::format("مدیریت ادمین‌ها", true) . "\n\n";
    
    if (empty($adminsList)) {
        $message .= HTMLParser::blockquote("هیچ ادمین اضافی ثبت نشده است");
    } else {
        $message .= "📋 لیست ادمین‌ها:\n\n";
        
        $users = Database::getAllUsers();
        foreach ($adminsList as $index => $adminId) {
            $adminInfo = null;
            foreach ($users as $user) {
                if ($user['user_id'] == $adminId) {
                    $adminInfo = $user;
                    break;
                }
            }
            
            if ($adminInfo) {
                $name = $adminInfo['first_name'];
                if (!empty($adminInfo['username'])) {
                    $name .= " (@" . $adminInfo['username'] . ")";
                }
                $message .= ($index + 1) . ". " . HTMLParser::format($name, false, false, true) . "\n";
                $message .= "   ID: " . HTMLParser::format($adminId, false, false, true) . "\n\n";
            } else {
                $message .= ($index + 1) . ". User ID: " . HTMLParser::format($adminId, false, false, true) . "\n\n";
            }
        }
    }
    
    $message .= "\n💡 برای افزودن ادمین جدید:\n";
    $message .= HTMLParser::format("/addadmin <user_id>", false, false, true) . "\n\n";
    $message .= "💡 برای حذف ادمین:\n";
    $message .= HTMLParser::format("/removeadmin <user_id>", false, false, true);
    
    $keyboard = Telegram::makeKeyboard([
        [
            ['text' => '➕ افزودن ادمین'],
            ['text' => '➖ حذف ادمین']
        ],
        [
            ['text' => '📋 لیست ادمین‌ها']
        ],
        [
            ['text' => '🔙 بازگشت']
        ]
    ]);
    
    Telegram::sendMessage($chatId, $message, $keyboard);
}

/**
 * راهنمای افزودن ادمین
 */
function handleAddAdminGuide($chatId, $userId) {
    if ($userId != ADMIN_ID) {
        Telegram::sendMessage($chatId, "⚠️ فقط ادمین اصلی می‌تواند ادمین اضافه کند!");
        return;
    }
    
    $message = "➕ " . HTMLParser::format("افزودن ادمین جدید", true) . "\n\n";
    $message .= HTMLParser::blockquote("User ID کاربر مورد نظر را ارسال کنید");
    $message .= "\n\n💡 نکته:\n";
    $message .= "• برای دریافت User ID، از دکمه 'جستجوی کاربر' استفاده کنید\n";
    $message .= "• یا از دستور /addadmin <user_id> استفاده کنید\n\n";
    $message .= "مثال: " . HTMLParser::format("/addadmin 123456789", false, false, true);
    
    Telegram::sendMessage($chatId, $message);
    Database::setUploadState($userId, 'awaiting_add_admin', []);
}

/**
 * افزودن ادمین جدید
 */
function handleAddAdmin($chatId, $userId, $newAdminId) {
    if ($userId != ADMIN_ID) {
        Telegram::sendMessage($chatId, "⚠️ فقط ادمین اصلی می‌تواند ادمین اضافه کند!");
        return;
    }
    
    if (!is_numeric($newAdminId)) {
        Telegram::sendMessage($chatId, "❌ User ID باید عدد باشد!");
        return;
    }
    
    $newAdminId = intval($newAdminId);
    
    // بررسی که ادمین اصلی نباشد
    if ($newAdminId == ADMIN_ID) {
        Telegram::sendMessage($chatId, "⚠️ این کاربر ادمین اصلی است!");
        return;
    }
    
    // بررسی وجود کاربر
    $user = Database::getUserById($newAdminId);
    if (!$user) {
        $message = "⚠️ این کاربر در ربات ثبت نام نکرده است!\n\n";
        $message .= "لطفاً ابتدا از کاربر بخواهید ربات را /start کند.";
        Telegram::sendMessage($chatId, $message);
        return;
    }
    
    $result = Database::addAdmin($newAdminId);
    
    if ($result) {
        $adminName = $user['first_name'];
        if (!empty($user['username'])) {
            $adminName .= " (@" . $user['username'] . ")";
        }
        
        $message = "✅ ادمین جدید با موفقیت اضافه شد!\n\n";
        $message .= "👤 نام: " . HTMLParser::format($adminName, true) . "\n";
        $message .= "🆔 User ID: " . HTMLParser::format($newAdminId, false, false, true);
        
        Telegram::sendMessage($chatId, $message, Telegram::getAdminKeyboard());
        
        // اطلاع به کاربر جدید
        $notifyMsg = "🎉 تبریک!\n\n";
        $notifyMsg .= "شما به عنوان " . HTMLParser::format("ادمین ربات", true) . " انتخاب شدید!\n\n";
        $notifyMsg .= "اکنون می‌توانید از دستور /panel برای دسترسی به پنل مدیریت استفاده کنید.";
        Telegram::sendMessage($newAdminId, $notifyMsg, Telegram::getAdminKeyboard());
    } else {
        Telegram::sendMessage($chatId, "⚠️ این کاربر قبلاً ادمین است!");
    }
    
    Database::clearUploadState($userId);
}

/**
 * راهنمای حذف ادمین
 */
function handleRemoveAdminGuide($chatId, $userId) {
    if ($userId != ADMIN_ID) {
        Telegram::sendMessage($chatId, "⚠️ فقط ادمین اصلی می‌تواند ادمین حذف کند!");
        return;
    }
    
    $adminsList = Database::getAdminsList();
    
    if (empty($adminsList)) {
        Telegram::sendMessage($chatId, "⚠️ هیچ ادمینی برای حذف وجود ندارد!");
        return;
    }
    
    $message = "➖ " . HTMLParser::format("حذف ادمین", true) . "\n\n";
    $message .= HTMLParser::blockquote("User ID ادمینی که می‌خواهید حذف کنید را ارسال کنید");
    $message .= "\n\n📋 لیست ادمین‌های فعلی:\n\n";
    
    $users = Database::getAllUsers();
    foreach ($adminsList as $index => $adminId) {
        $adminInfo = null;
        foreach ($users as $user) {
            if ($user['user_id'] == $adminId) {
                $adminInfo = $user;
                break;
            }
        }
        
        if ($adminInfo) {
            $name = $adminInfo['first_name'];
            if (!empty($adminInfo['username'])) {
                $name .= " (@" . $adminInfo['username'] . ")";
            }
            $message .= ($index + 1) . ". " . HTMLParser::format($name, false, false, true);
        } else {
            $message .= ($index + 1) . ". User ID: " . HTMLParser::format($adminId, false, false, true);
        }
        $message .= "\n   ID: " . HTMLParser::format($adminId, false, false, true) . "\n\n";
    }
    
    $message .= "\n💡 مثال:\n";
    $message .= HTMLParser::format("/removeadmin " . $adminsList[0], false, false, true);
    
    Telegram::sendMessage($chatId, $message);
    Database::setUploadState($userId, 'awaiting_remove_admin', []);
}

/**
 * حذف ادمین
 */
function handleRemoveAdmin($chatId, $userId, $adminIdToRemove) {
    if ($userId != ADMIN_ID) {
        Telegram::sendMessage($chatId, "⚠️ فقط ادمین اصلی می‌تواند ادمین حذف کند!");
        return;
    }
    
    if (!is_numeric($adminIdToRemove)) {
        Telegram::sendMessage($chatId, "❌ User ID باید عدد باشد!");
        return;
    }
    
    $adminIdToRemove = intval($adminIdToRemove);
    
    // بررسی که ادمین اصلی نباشد
    if ($adminIdToRemove == ADMIN_ID) {
        Telegram::sendMessage($chatId, "⚠️ نمی‌توانید ادمین اصلی را حذف کنید!");
        return;
    }
    
    $result = Database::removeAdmin($adminIdToRemove);
    
    if ($result) {
        $message = "✅ ادمین با موفقیت حذف شد!\n\n";
        $message .= "🆔 User ID: " . HTMLParser::format($adminIdToRemove, false, false, true);
        
        Telegram::sendMessage($chatId, $message, Telegram::getAdminKeyboard());
        
        // اطلاع به کاربر
        $notifyMsg = "⚠️ اطلاعیه\n\n";
        $notifyMsg .= "دسترسی ادمینی شما توسط مدیر اصلی " . HTMLParser::format("لغو", true) . " شد.\n\n";
        $notifyMsg .= "از این پس به پنل مدیریت دسترسی ندارید.";
        Telegram::sendMessage($adminIdToRemove, $notifyMsg, Telegram::getUserKeyboard());
    } else {
        Telegram::sendMessage($chatId, "⚠️ این کاربر ادمین نیست!");
    }
    
    Database::clearUploadState($userId);
}

/**
 * شروع فرآیند ارسال پیام همگانی
 */
function handleBroadcast($chatId, $userId) {
    $message = "📢 " . HTMLParser::format("پیام همگانی", true) . "\n\n";
    $message .= HTMLParser::blockquote("پیام یا فایل خود را برای ارسال به همه کاربران ارسال کنید");
    $message .= "\n\n💡 نکات:\n";
    $message .= "• می‌توانید متن، عکس، فایل، ویدیو یا هر نوع پیام دیگری ارسال کنید\n";
    $message .= "• برای فوروارد، پیام را از کانال یا چت دیگر فوروارد کنید\n";
    $message .= "• برای لغو عملیات، دستور /cancel را ارسال کنید";
    
    $keyboard = Telegram::makeKeyboard([
        [
            ['text' => '❌ لغو']
        ]
    ]);
    
    Telegram::sendMessage($chatId, $message, $keyboard);
    
    // تنظیم state برای دریافت پیام
    Database::setUploadState($userId, 'awaiting_broadcast_message', []);
}

/**
 * ارسال پیام همگانی به همه کاربران
 */
function sendBroadcast($chatId, $userId, $message) {
    $users = Database::getAllUsers();
    
    if (empty($users)) {
        Telegram::sendMessage($chatId, "⚠️ هیچ کاربری در ربات ثبت نشده است!", Telegram::getAdminKeyboard());
        Database::clearUploadState($userId);
        return;
    }
    
    $totalUsers = count($users);
    $successCount = 0;
    $failCount = 0;
    
    // ارسال پیام شروع
    $startMsg = "🔄 " . HTMLParser::format("در حال ارسال...", true) . "\n\n";
    $startMsg .= "تعداد کاربران: " . HTMLParser::format(number_format($totalUsers), true);
    $statusMessage = Telegram::sendMessage($chatId, $startMsg);
    $statusMessageId = $statusMessage['result']['message_id'] ?? null;
    
    // ارسال پیام به همه کاربران
    foreach ($users as $user) {
        $targetUserId = $user['user_id'];
        
        // عدم ارسال به خود ادمین
        if ($targetUserId == $userId) {
            continue;
        }
        
        // بررسی اینکه کاربر بلاک نیست
        if (isset($user['is_blocked']) && $user['is_blocked']) {
            $failCount++;
            continue;
        }
        
        try {
            $result = null;
            
            // بررسی نوع پیام و ارسال مناسب
            if (isset($message['forward_from']) || isset($message['forward_from_chat'])) {
                // پیام فوروارد شده
                $result = Telegram::forwardMessage($targetUserId, $chatId, $message['message_id']);
            } elseif (isset($message['text'])) {
                // پیام متنی
                $result = Telegram::sendMessage($targetUserId, $message['text']);
            } elseif (isset($message['photo'])) {
                // عکس
                $photo = end($message['photo']);
                $caption = $message['caption'] ?? null;
                $result = Telegram::sendPhoto($targetUserId, $photo['file_id'], $caption);
            } elseif (isset($message['video'])) {
                // ویدیو
                $caption = $message['caption'] ?? null;
                $result = Telegram::sendVideo($targetUserId, $message['video']['file_id'], $caption);
            } elseif (isset($message['document'])) {
                // فایل
                $caption = $message['caption'] ?? null;
                $result = Telegram::sendDocument($targetUserId, $message['document']['file_id'], $caption);
            } elseif (isset($message['audio'])) {
                // صوت
                $caption = $message['caption'] ?? null;
                $result = Telegram::sendAudio($targetUserId, $message['audio']['file_id'], $caption);
            } elseif (isset($message['voice'])) {
                // ویس
                $caption = $message['caption'] ?? null;
                $result = Telegram::sendVoice($targetUserId, $message['voice']['file_id'], $caption);
            }
            
            if ($result && isset($result['ok']) && $result['ok']) {
                $successCount++;
            } else {
                $failCount++;
            }
            
            // به‌روزرسانی پیام وضعیت هر 10 کاربر
            if (($successCount + $failCount) % 10 == 0 && $statusMessageId) {
                $progressMsg = "🔄 " . HTMLParser::format("در حال ارسال...", true) . "\n\n";
                $progressMsg .= "تعداد کل: " . HTMLParser::format(number_format($totalUsers), true) . "\n";
                $progressMsg .= "✅ موفق: " . HTMLParser::format(number_format($successCount), true) . "\n";
                $progressMsg .= "❌ ناموفق: " . HTMLParser::format(number_format($failCount), true);
                
                Telegram::request('editMessageText', [
                    'chat_id' => $chatId,
                    'message_id' => $statusMessageId,
                    'text' => $progressMsg,
                    'parse_mode' => 'HTML'
                ]);
            }
            
            // تاخیر کوتاه برای جلوگیری از فلود
            usleep(50000); // 50 میلی‌ثانیه
            
        } catch (Exception $e) {
            $failCount++;
        }
    }
    
    // پیام نهایی
    $finalMsg = "✅ " . HTMLParser::format("ارسال پیام همگانی تکمیل شد", true) . "\n\n";
    $finalMsg .= "📊 گزارش:\n";
    $finalMsg .= "• تعداد کل کاربران: " . HTMLParser::format(number_format($totalUsers), true) . "\n";
    $finalMsg .= "• ارسال موفق: " . HTMLParser::format(number_format($successCount), true) . "\n";
    $finalMsg .= "• ارسال ناموفق: " . HTMLParser::format(number_format($failCount), true) . "\n\n";
    
    $percentage = $totalUsers > 0 ? round(($successCount / $totalUsers) * 100, 1) : 0;
    $finalMsg .= "📈 نرخ موفقیت: " . HTMLParser::format($percentage . "%", true);
    
    if ($statusMessageId) {
        Telegram::request('editMessageText', [
            'chat_id' => $chatId,
            'message_id' => $statusMessageId,
            'text' => $finalMsg,
            'parse_mode' => 'HTML'
        ]);
    } else {
        Telegram::sendMessage($chatId, $finalMsg);
    }
    
    // نمایش پنل ادمین
    Telegram::sendMessage($chatId, "🎛 پنل مدیریت:", Telegram::getAdminKeyboard());
    
    // پاک کردن state
    Database::clearUploadState($userId);
}

/**
 * تغییر حالت تعمیر و نگهداری
 */
function toggleMaintenance($chatId) {
    $settings = Database::getSettings();
    $newStatus = !$settings['maintenance_mode'];
    
    Database::updateSetting('maintenance_mode', $newStatus);
    
    $statusText = $newStatus ? 'فعال' : 'غیرفعال';
    $emoji = $newStatus ? '🔴' : '🟢';
    
    $message = "{$emoji} حالت تعمیر و نگهداری " . HTMLParser::format($statusText, true) . " شد!\n\n";
    
    if ($newStatus) {
        $message .= HTMLParser::blockquote("ربات برای کاربران عادی غیرفعال است");
    } else {
        $message .= HTMLParser::blockquote("ربات برای همه فعال است");
    }
    
    Telegram::sendMessage($chatId, $message, Telegram::getSettingsKeyboard());
}

/**
 * تغییر وضعیت جوین اجباری
 */
function toggleForceJoin($chatId) {
    $settings = Database::getSettings();
    $newStatus = !$settings['force_join'];
    
    Database::updateSetting('force_join', $newStatus);
    
    $statusText = $newStatus ? 'فعال' : 'غیرفعال';
    $emoji = $newStatus ? '🟢' : '🔴';
    
    $message = "{$emoji} جوین اجباری " . HTMLParser::format($statusText, true) . " شد!\n\n";
    
    $channels = Database::getForceJoinChannels();
    
    if ($newStatus && empty($channels)) {
        $message .= "⚠️ " . HTMLParser::format("توجه:", true) . " هیچ کانالی ثبت نشده است.\n\n";
        $message .= HTMLParser::blockquote("از بخش 📢 مدیریت کانال‌ها برای افزودن کانال استفاده کنید");
    } else if ($newStatus) {
        $message .= "تعداد کانال‌های ثبت شده: " . HTMLParser::format(count($channels), true) . "\n\n";
        $message .= HTMLParser::blockquote("کاربران باید عضو تمام کانال‌ها باشند");
    } else {
        $message .= HTMLParser::blockquote("عضویت در کانال الزامی نیست");
    }
    
    Telegram::sendMessage($chatId, $message, Telegram::getSettingsKeyboard());
}

/**
 * تنظیم کانال الزامی
 */
function setForceJoinChannel($chatId, $channelUsername, $channelId = null) {
    // این تابع deprecated شده - از handleAddChannel استفاده کنید
    handleAddChannel($chatId, $channelUsername);
}

/**
 * نمایش منوی انتخاب نوع آپلود
 */
function showUploadMenu($chatId) {
    $message = "📤 " . HTMLParser::format("آپلود فایل", true) . "\n\n";
    $message .= HTMLParser::blockquote("لطفاً نوع آپلود را انتخاب کنید");
    $message .= "\n\n🔹 " . HTMLParser::format("آپلود از تلگرام:", true) . " فایل را مستقیماً ارسال کنید\n";
    $message .= "🔹 " . HTMLParser::format("آپلود از لینک:", true) . " لینک دانلود فایل را ارسال کنید";
    
    Telegram::sendMessage($chatId, $message, Telegram::getUploadTypeKeyboard());
}

/**
 * نمایش راهنمای پنل ادمین
 */
function handleGuide($chatId) {
    $message = "📋 " . HTMLParser::format("راهنمای پنل ادمین", true) . "\n\n";
    
    $message .= HTMLParser::blockquote("دستورات و راهنمای استفاده") . "\n\n";
    
    $message .= "📊 " . HTMLParser::format("آمار کاربران", true) . "\n";
    $message .= "مشاهده تعداد کاربران، فایل‌ها و دانلودها\n\n";
    
    $message .= "📁 " . HTMLParser::format("فایل‌های من", true) . "\n";
    $message .= "لیست تمام فایل‌های آپلود شده\n";
    $message .= "دستور: /list یا /files\n\n";
    
    $message .= "📤 " . HTMLParser::format("آپلود فایل", true) . "\n";
    $message .= "فقط فایل را ارسال کنید\n\n";
    
    $message .= "🖥 " . HTMLParser::format("وضعیت سرور", true) . "\n";
    $message .= "مشاهده وضعیت و منابع سرور\n\n";
    
    $message .= "⚙️ " . HTMLParser::format("تنظیمات", true) . "\n";
    $message .= "مدیریت حالت تعمیر و جوین اجباری\n\n";
    
    $message .= HTMLParser::format("دستورات اضافی:", true) . "\n";
    $message .= "/edit <file_id> - ویرایش فایل\n";
    $message .= "/delete <id> - حذف فایل\n";
    $message .= "/setchannel @username - تنظیم کانال\n";
    $message .= "/broadcast پیام - ارسال پیام همگانی\n";
    
    Telegram::sendMessage($chatId, $message, Telegram::getAdminKeyboard());
}

/**
 * نمایش منوی مدیریت فایل
 */
function showFileManageMenu($chatId, $fileId) {
    $file = Database::getFile($fileId);
    
    if (!$file) {
        Telegram::sendMessage($chatId, "❌ فایل یافت نشد!");
        return;
    }
    
    $message = "🗂 " . HTMLParser::format("مدیریت فایل", true) . "\n\n";
    $message .= "📁 " . HTMLParser::format($file['file_name'], true) . "\n";
    $message .= "💾 حجم: " . HTMLParser::formatFileSize($file['file_size']) . "\n";
    $message .= "📥 دانلودها: " . number_format($file['download_count']) . "\n";
    
    $statusEmoji = $file['is_active'] ? "🟢" : "🔴";
    $statusText = $file['is_active'] ? "فعال" : "غیرفعال";
    $message .= "📊 وضعیت: {$statusEmoji} {$statusText}\n";
    
    if (!empty($file['password'])) {
        $message .= "🔐 رمز: دارد\n";
    }
    
    if (!empty($file['expire_date'])) {
        $message .= "⏰ انقضا: " . date('Y/m/d H:i', strtotime($file['expire_date'])) . "\n";
    }
    
    if ($file['download_limit'] > 0) {
        $message .= "🔢 محدودیت: " . $file['download_limit'] . " دانلود\n";
    }
    
    $message .= "\n" . HTMLParser::blockquote("از دکمه‌های زیر برای مدیریت استفاده کنید");
    
    $keyboard = [
        [
            ['text' => ($file['is_active'] ? '🔴 غیرفعال کردن' : '🟢 فعال کردن'), 'callback_data' => 'file_toggle_' . $fileId]
        ],
        [
            ['text' => '🔐 تنظیم رمز', 'callback_data' => 'file_setpass_' . $fileId],
            ['text' => '❌ حذف رمز', 'callback_data' => 'file_delpass_' . $fileId]
        ],
        [
            ['text' => '⏰ تنظیم انقضا', 'callback_data' => 'file_expire_' . $fileId],
            ['text' => '🔢 محدودیت دانلود', 'callback_data' => 'file_limit_' . $fileId]
        ],
        [
            ['text' => '📝 ویرایش تایتل', 'callback_data' => 'file_title_' . $fileId],
            ['text' => '📄 ویرایش توضیحات', 'callback_data' => 'file_caption_' . $fileId]
        ],
        [
            ['text' => '✏️ تغییر نام', 'callback_data' => 'file_rename_' . $fileId]
        ],
        [
            ['text' => '🗑 حذف فایل', 'callback_data' => 'file_delete_' . $fileId]
        ],
        [
            ['text' => '🔙 بازگشت', 'callback_data' => 'file_close']
        ]
    ];
    
    Telegram::sendMessage($chatId, $message, Telegram::makeInlineKeyboard($keyboard));
}

/**
 * پردازش callback های مدیریت فایل
 */
function handleFileManageCallback($chatId, $userId, $data, $messageId) {
    $parts = explode('_', $data);
    $action = $parts[1] ?? '';
    $fileId = $parts[2] ?? '';
    
    switch ($action) {
        case 'toggle':
            toggleFileStatus($chatId, $fileId, $messageId);
            break;
            
        case 'setpass':
            startSetPassword($chatId, $userId, $fileId, $messageId);
            break;
            
        case 'delpass':
            deleteFilePassword($chatId, $fileId, $messageId);
            break;
            
        case 'expire':
            startSetExpire($chatId, $userId, $fileId, $messageId);
            break;
            
        case 'limit':
            startSetLimit($chatId, $userId, $fileId, $messageId);
            break;
            
        case 'rename':
            startRenameFile($chatId, $userId, $fileId, $messageId);
            break;
            
        case 'title':
            startEditTitle($chatId, $userId, $fileId, $messageId);
            break;
            
        case 'caption':
            startEditCaption($chatId, $userId, $fileId, $messageId);
            break;
            
        case 'delete':
            confirmDeleteFile($chatId, $fileId, $messageId);
            break;
            
        case 'confirmdelete':
            executeDeleteFile($chatId, $fileId, $messageId);
            break;
            
        case 'close':
            Telegram::request('deleteMessage', [
                'chat_id' => $chatId,
                'message_id' => $messageId
            ]);
            break;
    }
}

/**
 * تغییر وضعیت فایل (فعال/غیرفعال)
 */
function toggleFileStatus($chatId, $fileId, $messageId) {
    $file = Database::getFile($fileId);
    
    if (!$file) {
        Telegram::sendMessage($chatId, "❌ فایل یافت نشد!");
        return;
    }
    
    $newStatus = !$file['is_active'];
    Database::updateFile($fileId, ['is_active' => $newStatus]);
    
    $statusText = $newStatus ? "فعال" : "غیرفعال";
    Telegram::sendMessage($chatId, "✅ وضعیت فایل به {$statusText} تغییر کرد.");
    
    // به‌روزرسانی منو
    showFileManageMenu($chatId, $fileId);
    
    Telegram::request('deleteMessage', [
        'chat_id' => $chatId,
        'message_id' => $messageId
    ]);
}

/**
 * شروع تنظیم رمز
 */
function startSetPassword($chatId, $userId, $fileId, $messageId) {
    Database::setUploadState($userId, 'file_set_password', ['file_id' => $fileId]);
    
    Telegram::sendMessage($chatId, "🔐 رمز جدید برای فایل را ارسال کنید:\n\n" . HTMLParser::blockquote("برای لغو /cancel ارسال کنید"));
    
    Telegram::request('deleteMessage', [
        'chat_id' => $chatId,
        'message_id' => $messageId
    ]);
}

/**
 * حذف رمز فایل
 */
function deleteFilePassword($chatId, $fileId, $messageId) {
    Database::updateFile($fileId, ['password' => '']);
    
    Telegram::sendMessage($chatId, "✅ رمز فایل حذف شد.");
    
    showFileManageMenu($chatId, $fileId);
    
    Telegram::request('deleteMessage', [
        'chat_id' => $chatId,
        'message_id' => $messageId
    ]);
}

/**
 * شروع تنظیم تاریخ انقضا
 */
function startSetExpire($chatId, $userId, $fileId, $messageId) {
    Database::setUploadState($userId, 'file_set_expire', ['file_id' => $fileId]);
    
    $message = "⏰ تعداد روز تا انقضا را ارسال کنید:\n\n";
    $message .= "مثال: 7 (برای 7 روز)\n";
    $message .= "یا 0 برای حذف انقضا\n\n";
    $message .= HTMLParser::blockquote("برای لغو /cancel ارسال کنید");
    
    Telegram::sendMessage($chatId, $message);
    
    Telegram::request('deleteMessage', [
        'chat_id' => $chatId,
        'message_id' => $messageId
    ]);
}

/**
 * شروع تنظیم محدودیت دانلود
 */
function startSetLimit($chatId, $userId, $fileId, $messageId) {
    Database::setUploadState($userId, 'file_set_limit', ['file_id' => $fileId]);
    
    $message = "🔢 محدودیت دانلود را ارسال کنید:\n\n";
    $message .= "مثال: 100 (برای 100 بار دانلود)\n";
    $message .= "یا 0 برای نامحدود\n\n";
    $message .= HTMLParser::blockquote("برای لغو /cancel ارسال کنید");
    
    Telegram::sendMessage($chatId, $message);
    
    Telegram::request('deleteMessage', [
        'chat_id' => $chatId,
        'message_id' => $messageId
    ]);
}

/**
 * شروع تغییر نام فایل
 */
function startRenameFile($chatId, $userId, $fileId, $messageId) {
    Database::setUploadState($userId, 'file_rename', ['file_id' => $fileId]);
    
    Telegram::sendMessage($chatId, "✏️ نام جدید فایل را ارسال کنید:\n\n" . HTMLParser::blockquote("برای لغو /cancel ارسال کنید"));
    
    Telegram::request('deleteMessage', [
        'chat_id' => $chatId,
        'message_id' => $messageId
    ]);
}

/**
 * شروع ویرایش تایتل
 */
function startEditTitle($chatId, $userId, $fileId, $messageId) {
    Database::setUploadState($userId, 'file_edit_title', ['file_id' => $fileId]);
    
    Telegram::sendMessage($chatId, "📝 تایتل جدید فایل را ارسال کنید:\n\n" . HTMLParser::blockquote("برای حذف تایتل، - ارسال کنید\nبرای لغو /cancel ارسال کنید"));
    
    Telegram::request('deleteMessage', [
        'chat_id' => $chatId,
        'message_id' => $messageId
    ]);
}

/**
 * شروع ویرایش توضیحات
 */
function startEditCaption($chatId, $userId, $fileId, $messageId) {
    Database::setUploadState($userId, 'file_edit_caption', ['file_id' => $fileId]);
    
    Telegram::sendMessage($chatId, "📄 توضیحات جدید فایل را ارسال کنید:\n\n" . HTMLParser::blockquote("برای حذف توضیحات، - ارسال کنید\nبرای لغو /cancel ارسال کنید"));
    
    Telegram::request('deleteMessage', [
        'chat_id' => $chatId,
        'message_id' => $messageId
    ]);
}

/**
 * تایید حذف فایل
 */
function confirmDeleteFile($chatId, $fileId, $messageId) {
    $keyboard = [
        [
            ['text' => '✅ بله، حذف شود', 'callback_data' => 'file_confirmdelete_' . $fileId],
            ['text' => '❌ انصراف', 'callback_data' => 'file_close']
        ]
    ];
    
    Telegram::request('editMessageText', [
        'chat_id' => $chatId,
        'message_id' => $messageId,
        'text' => "⚠️ آیا از حذف این فایل اطمینان دارید؟\n\n" . HTMLParser::blockquote("این عملیات غیرقابل بازگشت است!"),
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode(['inline_keyboard' => $keyboard])
    ]);
}

/**
 * حذف نهایی فایل
 */
function executeDeleteFile($chatId, $fileId, $messageId) {
    $result = Database::deleteFile($fileId);
    
    if ($result) {
        Telegram::request('editMessageText', [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => "✅ فایل با موفقیت حذف شد.",
            'parse_mode' => 'HTML'
        ]);
    } else {
        Telegram::sendMessage($chatId, "❌ خطا در حذف فایل!");
    }
}

/**
 * پردازش ورودی‌های مربوط به مدیریت فایل
 */
function processFileManagementInput($chatId, $userId, $text) {
    $state = Database::getUploadState($userId);
    
    if (!$state || !isset($state['data']['file_id'])) {
        return false;
    }
    
    $fileId = $state['data']['file_id'];
    
    if ($text == '/cancel') {
        Database::clearUploadState($userId);
        Telegram::sendMessage($chatId, "❌ عملیات لغو شد.");
        showFileManageMenu($chatId, $fileId);
        return true;
    }
    
    switch ($state['state']) {
        case 'file_set_password':
            Database::updateFile($fileId, ['password' => $text]);
            Database::clearUploadState($userId);
            Telegram::sendMessage($chatId, "✅ رمز فایل تنظیم شد: " . $text);
            showFileManageMenu($chatId, $fileId);
            return true;
            
        case 'file_set_expire':
            if (is_numeric($text)) {
                $days = intval($text);
                $expireDate = $days > 0 ? date('Y-m-d H:i:s', strtotime("+{$days} days")) : null;
                Database::updateFile($fileId, ['expire_date' => $expireDate]);
                Database::clearUploadState($userId);
                
                $msg = $days > 0 ? "✅ تاریخ انقضا به {$days} روز دیگر تنظیم شد." : "✅ تاریخ انقضا حذف شد.";
                Telegram::sendMessage($chatId, $msg);
                showFileManageMenu($chatId, $fileId);
            } else {
                Telegram::sendMessage($chatId, "❌ لطفا یک عدد ارسال کنید!");
            }
            return true;
            
        case 'file_set_limit':
            if (is_numeric($text)) {
                $limit = intval($text);
                Database::updateFile($fileId, ['download_limit' => $limit]);
                Database::clearUploadState($userId);
                
                $msg = $limit > 0 ? "✅ محدودیت دانلود به {$limit} بار تنظیم شد." : "✅ محدودیت دانلود حذف شد.";
                Telegram::sendMessage($chatId, $msg);
                showFileManageMenu($chatId, $fileId);
            } else {
                Telegram::sendMessage($chatId, "❌ لطفا یک عدد ارسال کنید!");
            }
            return true;
            
        case 'file_rename':
            Database::updateFile($fileId, ['file_name' => $text]);
            Database::clearUploadState($userId);
            Telegram::sendMessage($chatId, "✅ نام فایل تغییر کرد: " . $text);
            showFileManageMenu($chatId, $fileId);
            return true;
            
        case 'file_edit_title':
            $newTitle = ($text === '-') ? '' : $text;
            Database::updateFile($fileId, ['title' => $newTitle]);
            Database::clearUploadState($userId);
            
            $msg = ($text === '-') ? "✅ تایتل فایل حذف شد." : "✅ تایتل فایل تغییر کرد: " . $text;
            Telegram::sendMessage($chatId, $msg);
            showFileManageMenu($chatId, $fileId);
            return true;
            
        case 'file_edit_caption':
            $newCaption = ($text === '-') ? '' : $text;
            Database::updateFile($fileId, ['caption' => $newCaption]);
            Database::clearUploadState($userId);
            
            $msg = ($text === '-') ? "✅ توضیحات فایل حذف شد." : "✅ توضیحات فایل تغییر کرد.";
            Telegram::sendMessage($chatId, $msg);
            showFileManageMenu($chatId, $fileId);
            return true;
    }
    
    return false;
}
